/* Started Dec 5, 1994 at Xerox.
 * All wrongs reversed. */
#include <errno.h>
#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>

#include <pwd.h>
#include <grp.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifdef __USLC__
#include <sys/mkdev.h>
#endif
#if defined(__linux__) || defined(__svr4__)
#include <sys/sysmacros.h>
#endif

#include "opts.h"
#include "version.h"

/* output:
 *	File: foo[/@*!][ -> bar]
 *	Size: %-8d			Blocks: %-8d
 *	Owner: <%5d (%s)>18s		Group: <%5d (%s)>18s
 *	Mode: 04777 (-rwsrwxrwx)	Type: ...
 *	Inode: ...			Device: ...
 */

enum types {
	T_DIR, T_LNK, T_EXE, T_FIFO, T_BLK, T_CHR, T_SOCK, T_MAGIC, T_PLAIN
};

/* This is to support later work with making the output optional, and
 * supporting usages like 'stat -lq file' to succeed if file is a link.
 */
typedef struct {
	int quiet, blocks, character, block, directory, fifo,
		socket, magic, symlink, owner, group, mode;
	uid_t ownerval;
	gid_t groupval;
	mode_t modeval;
	enum types typeval;
} statquery;

int prbuf(char *name, struct stat *buf, statquery *sq);
int ckbuf(char *name, struct stat *buf, statquery *sq);

char typechars[] = "/@*++++!";
char typelets[]  = "dl-fbcs!-";
char *typenames[] = {
	"directory",
	"symbolic link",
	"executable",
	"FIFO",
	"block special",
	"character special",
	"socket",
	"unknown(!)",
	"plain"
};

/* return 1 if buf matches statquery */
int
ckbuf(char *name, struct stat *buf, statquery *sq) {
	mode_t mode = buf->st_mode;
/*	enum types type = T_MAGIC; */

	if (sq->directory && !S_ISDIR(mode))
		return 1;
	/* XXX Fill this in */

	return 0;
}

int
prbuf(char *name, struct stat *buf, statquery *sq) {
	mode_t mode = buf->st_mode;
	char modechars[11] = "----------";
	enum types type = T_MAGIC;
	int isdev = 0;

	if (S_ISFIFO(mode))
		type = T_FIFO;
	else if ((mode & S_IFLNK) == S_IFLNK)
		type = T_LNK;
	else if (S_ISBLK(mode))
		type = T_BLK;
	else if (S_ISCHR(mode))
		type = T_CHR;
	else if (S_ISDIR(mode))
		type = T_DIR;
	else if ((mode & S_IFSOCK) == S_IFSOCK)
		type = T_SOCK;
	else if (S_ISREG(mode)) {
		if (mode & 0111)
			type = T_EXE;
		else
			type = T_PLAIN;
	}

	if (type == T_BLK || type == T_CHR)
		isdev = 1;

	printf("File:   %s%c", name, typechars[type]);
	if (type == T_LNK) {
		char *foo = NULL;
		int test = 16, n;
		do {
			test += 16;
			free(foo);
			foo = malloc(test);
		} while ((n = readlink(name, foo, test)) == test);

		if (n == -1)
			strcpy(foo, "<unknown>");
		else
			foo[n] = '\0';
		printf(" -> %s", foo);
	}
	putchar('\n');
	if (isdev) {
		printf("Major:  %-8lu                   Minor:   %-8lu\n",
			(unsigned long) major(buf->st_rdev),
			(unsigned long) minor(buf->st_rdev));
	} else {
		printf("Size:   %-8lu                   Blocks:  %-8lu\n",
			(unsigned long) buf->st_size,
			(unsigned long) buf->st_blocks);
	}

	/* Get owner and group info */
	{
		char *oname, *gname;
		struct passwd *pwbuf = getpwuid(buf->st_uid);
		struct group *grbuf = getgrgid(buf->st_gid);

		if (pwbuf)
			oname = pwbuf->pw_name;
		else
			oname = "<unknown>";
		
		if (grbuf)
			gname = grbuf->gr_name;
		else
			gname = "<unknown>";


		printf("Owner:  %-5d (%s)%*sGroup:   %-5d (%s)\n",
			(int) buf->st_uid, oname, 
			(int) (19 - strlen(oname)), "",
			(int) buf->st_gid, gname);
	}

	/* Handle mode bits */
	{
		modechars[0] = typelets[type];

		if (mode & S_IRUSR)
			modechars[1] = 'r';
		if (mode & S_IWUSR)
			modechars[2] = 'w';
		if (mode & S_IXUSR)
			modechars[3] = mode & S_ISUID ? 's' : 'x';
		else if (mode & S_ISUID)
			modechars[3] = 'S';

		if (mode & S_IRGRP)
			modechars[4] = 'r';
		if (mode & S_IWGRP)
			modechars[5] = 'w';
		if (mode & S_IXGRP)
			modechars[6] = mode & S_ISGID ? 's' : 'x';
		else if (mode & S_ISGID)
			modechars[6] = 'S';

		if (mode & S_IROTH)
			modechars[7] = 'r';
		if (mode & S_IWOTH)
			modechars[8] = 'w';
		if (mode & S_IXOTH)
			modechars[9] = mode & S_ISVTX ? 't' : 'x';
		else if (mode & S_ISVTX)
			modechars[9] = 'T';
	}

	printf("Mode:   %04o  (%s)         Type:    %s\n",
		(unsigned) mode & ~S_IFMT, modechars,
		typenames[type]);
	printf("Inode:  %-7ld                    Device:  %-7ld\n",
		(long) buf->st_ino, (long) buf->st_dev);
	printf("Links:  %d\n", (int) buf->st_nlink);
	printf("Last access: %s", ctime(&buf->st_atime));
	printf("Last modify: %s", ctime(&buf->st_mtime));
	printf("Last change: %s", ctime(&buf->st_ctime));
	return 0;
}

int
main(int argc, char *argv[]) {
	struct stat foo;
	statquery sq = {
		1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0
	};
	int (*funcp)(const char *, struct stat *) = lstat;
	int (*ckpr)(char *, struct stat *, statquery *) = prbuf;
	int retval = 0;
	int c, i;

	while ((c = optsopt(argc, argv, "lqV")) != -1) {
		switch (c) {
			case 'l':
				funcp = stat;
				break;
			case 'q':
				ckpr = ckbuf;
				break;
			case 'V':
				version();
				break;
			default:
				fprintf(stderr, "usage: stat [-lqV] [file ...]\n");
				exit(1);
				break;
		}
	}

	if (optsind < argc) {
		for (i = optsind; i < argc; ++i) {
			memset(&foo, 0, sizeof(foo));
			if (!strcmp(argv[i], "-")) {
				if (!(retval = fstat(0, &foo)))
					retval = (*ckpr)("<stdin>", &foo, &sq);
				else
					fprintf(stderr, "stat: %s\n",
						strerror(errno));
			} else if (!(retval = (*funcp)(argv[i], &foo))) {
				retval = (*ckpr)(argv[i], &foo, &sq);
			} else {
				retval = 1;
				fprintf(stderr, "stat: '%s': %s\n",
					argv[i], strerror(errno));
			}
			if (i + 1 < argc)
				putchar('\n');
		}
	} else {
		memset(&foo, 0, sizeof(foo));
		if (!(retval = fstat(0, &foo))) {
			retval = (*ckpr)("<stdin>", &foo, &sq);
		} else {
			retval = 1;
			fprintf(stderr, "stat: %s\n", strerror(errno));
		}
	}

	return retval ? EXIT_FAILURE : EXIT_SUCCESS;
}
