/* Copyright 1991-1998 Peter Seebach.  All rights reserved.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>

static void roll(int, int, int, int, int, int, int);
static void parse(char *);

static char *macros[26];
static char lower[] = "abcdefghijklmnopqrstuvwxyz";
struct bigmac { char *name, *value; };
struct bigmac *bigmacs;
int nmacs;

static int
rnd(long n)
{
	return n < 2 ? 0 : random() % n;
}

static int
intcmp(const void *a, const void *b)
{
	int n = *(int *) a, m = *(int *) b;
	if (n < m)
		return -1;
	return n > m;
}

static void
addmac(char *name, char *value) {
	struct bigmac *new;
	if (value && strlen(value)) {
		new = malloc((nmacs + 2) * sizeof(struct bigmac));
		if (bigmacs) {
			struct bigmac *b;
			for (b = bigmacs; b->name; ++b) {
				if (!strcmp(b->name, name)) {
					free(b->value);
					b->value = strdup(value);
					free(new);
					return;
				}
			}
			/* if we get here, we're adding one */
			memcpy(new, bigmacs, nmacs * sizeof(struct bigmac));
			free(bigmacs);
		}
		new[nmacs].name = strdup(name);
		new[nmacs].value = strdup(value);
		++nmacs;
		new[nmacs].name = NULL;
		new[nmacs].value = NULL;
	} else {
		struct bigmac *b, *n;
		new = malloc((nmacs + 1) * sizeof(struct bigmac));
		n = new;
		for (b = bigmacs; b->name; ++b) {
			if (strcmp(b->name, name)) {
				n->name = b->name;
				n->value = b->value;
				++n;
			} else {
				--nmacs;
			}
		}
		n->name = NULL;
		n->value = NULL;
		free(bigmacs);
	}
	bigmacs = new;
}

void
parse(char *s)
{
	FILE *fp;
	char temp[1024];
	char *t, *u;
	static int total = 1, omit = 0, size = 6, dice = 3, times = 1;
	static int plus = 0, average;
	int i;
	static char savebuf[80] = "3d6x6t";

	if (!s)
		return;

	t = strtok(s, ",");
	do {
		if (t > s)
			printf("\n");

		while (t && isspace(*t))
			++t;

		if (t) {
			if (isupper(t[0])) {
				sprintf(savebuf, "<%c", tolower(t[0]));
				t = savebuf;
			}
			if (bigmacs) {
				struct bigmac *b;
				for (b = bigmacs; b->name; ++b) {
					if (!strcmp(b->name, t)) {
						t = b->value;
					}
				}
			}
			switch (*t) {
			case '!':
				system(t + 1);
				continue;
			case '?':
			case 'h':
				puts("Usage:");
				puts("n: roll n dice\t\t\tt: print only totals");
				puts("dn: use n-sided dice\t\txn: roll n times");
				puts("on: omit lowest n\t\tOn: omit highest n");
				puts("<a: run macro a\t\t\t>a: save macro a");
				puts("s: show macros\t\t\th, ?: print this message");
				puts("a: print average\t\tA: print only average");
				puts("T: print totals also");
				puts("[A-Z]: run macro [a-z]");
				puts("!s: run command s through shell");
				puts("l[filename]: load file, or ~/.dicerc");
				puts("w[filename]: write file, or ~/.dicerc");
				puts("");
				puts("commands may be comma-seperated.");
				puts("");
				puts("Examples:");
				puts("3d6x6t:\t\troll 3d6 6 times, provide totals.");
				puts("4d6x6o1t:\troll 4d6, drop lowest, 6 times, provide totals.");
				puts("4d6x1250o1A:\treport average of 1250 rolls of 4d6, drop lowest.");
				puts("");
				continue;
			case '>':
				if (strlen(t + 1) > 1) {
					if (t[1] == '!') {
						++t;
						addmac(t + 1, 0);
						printf("Cleared macro %s.\n", t + 1);
					} else {
						addmac(t + 1, savebuf);
						printf("Saved macro %s.\n", t + 1);
					}
				} else if (islower(t[1]) && strchr(lower, t[1])) {
					i = strchr(lower, t[1]) - lower;
					free(macros[i]);
					macros[i] = malloc(strlen(t + 2) + 1);
					if (macros[i]) {
						strcpy(macros[i],
							t[2] ? t + 2 : savebuf);
						printf("Saved macro %c.\n", t[1]);
					} else {
						printf("no memory for macro %c.\n", t[1]);
					}
				} else {
					printf("\"%s\": > requires a known lowercase argument.\n", t);
				}
				continue;
			case '<':
				if (islower(t[1]) && strchr(lower, t[1])) {
					i = strchr(lower, t[1]) - lower;
					if (macros[i]) {
						t = macros[i];
					} else {
						printf("no macro %c.\n",
							t[1]);
					}
					break;
				} else {
					printf("\"%s\": < requires a lowercase argument.\n", t);
					continue;
				}
			case 'l':
				if (t[1])
					strcpy(temp, t + 1);
				else
					sprintf(temp, "%s/.dicerc",
						(u = getenv("HOME")) ? u : "/tmp");
				fp = fopen(temp, "r");
				if (!fp) {
					fprintf(stderr, "warning: couldn't open file '%s'\n", temp);
					continue;
				}
				for (i = 0; i < 26; ++i) {
					char buf[1024];
					if (!fgets(buf, 1024, fp)) {
						fprintf(stderr, "warning: couldn't load macro %c.\n", lower[i]);
						break;
					} else {
						if (buf[strlen(buf) - 1] == '\n')
							buf[strlen(buf) - 1] = '\0';
						macros[i] = malloc(strlen(buf) + 1);
						if (!macros[i]) {
							printf("no memory to load macro %c.\n",
								lower[i]);
						} else {
							strcpy(macros[i], buf);
						}
					}
				}
				{
					char buf[1024];
					while (fgets(buf, 1024, fp)) {
						char *s, *t, *u;
						s = buf;
						t = strchr(buf, ' ');
						*t++ = '\0';
						u = t;
						while (*u && !isspace(*u))
							++u;
						*u = '\0';

						addmac(s, t);
					}
				}
				fprintf(stderr, "loaded macro file %s\n", temp);
				fclose(fp);
				continue;
			case 'w':
				if (t[1])
					strcpy(temp, t + 1);
				else
					sprintf(temp, "%s/.dicerc",
						(u = getenv("HOME")) ? u : "/tmp");
				fp = fopen(temp, "w");
				if (!fp) {
					fprintf(stderr, "warning: couldn't open file '%s'\n", temp);
					continue;
				}
				for (i = 0; i < 26; ++i) {
					u = macros[i];
					if (u) {
						if (fprintf(fp, "%s\n", u) != strlen(u) + 1) {
							fprintf(stderr, "warning: couldn't write macro %c\n", lower[i]);
							break;
						}
					} else {
						fputc('\n', fp);
					}
				}
				if (bigmacs) {
					struct bigmac *b = bigmacs;
					while (b->name) {
						fprintf(fp, "%s %s\n",
							b->name, b->value);
						++b;
					}
				}
				fprintf(stderr, "wrote macro file %s\n", temp);
				fclose(fp);
				continue;
			case 's':
				for (i = 0; i < 13; ++i) {
					printf("%c: %-30s %c: %-30s\n",
					 lower[i],
					 macros[i] ? macros[i] : "",
					 lower[i + 13],
					 macros[i + 13] ?  macros[i + 13] : "");
				}
				if (bigmacs) {
					struct bigmac *b = bigmacs;
					while (b->name) {
						printf("%s: %s\n",
							b->name, b->value);
						++b;
					}
				}
				continue;
			case 'r':
			case 0:
				t = savebuf;
				break;
			}	/* end switch(*t) */
		} else {
			t = savebuf;
			printf("%s\n", savebuf);
		}

		/*
		 * if t == savebuf, we're looking at the last macro,
		 * otherwise, we've entered a command - possibly a macro.
		 */
		if (t != savebuf)
			strcpy(savebuf, t);

		if (*t == 'q') {
			exit(0);
		}
		if ((u = strchr(t, 'd')))
			size = atoi(u + 1);
		/* otherwise, it defaults to what you used last */
		dice = (isdigit(*t)) ? atoi(t) : 1;
		omit = (u = strchr(t, 'o')) ? atoi(u + 1) : 0;
		if ((u = strchr(t, 'O')))
			omit = -atoi(u + 1);
		times = (u = strchr(t, 'x')) ? atoi(u + 1) : 1;
		times = (u = strchr(t, '*')) ? atoi(u + 1) : 1;
		total = !!strchr(t, 't');
		if (strchr(t, 'T'))
			total = -1;
		average = !!strchr(t, 'a');
		if ((u = strchr(t, '+')) || (u = strchr(t, '-')))
			plus = atoi(u);
		else
			plus = 0;
		if (strchr(t, 'A'))
			average = -1;

		if (abs(omit) >= dice) {
			fprintf(stderr, "error: can't omit %d dice; only %d rolled.\n",
				abs(omit), dice);
			continue;
		}
		if (plus && !total)
			total = -1;
		roll(dice, size, times, total, omit, average, plus);
	} while ((t = strtok(NULL, ",")));
}

void
roll(int dice, int size, int times, int total, int omit, int average, int plus)
{
	int i, j, x, n, *results;
	results = calloc(dice, sizeof(int));

	if (!results) {
		fprintf(stderr, "error: couldn't allocate space for results.\n");
		return;

	}
	x = 0;
	for (j = 0; j < times; ++j) {
		for (i = 0; i < dice; ++i) {
			results[i] = 1 + rnd(size);
		}
		if (omit) {
			qsort(results, dice, sizeof(int), intcmp);
			if (omit < 0)
				for (i = dice; i > (dice + omit); --i)
					results[i - 1] = 0;
			else
				for (i = 0; i < omit; ++i)
					results[i] = 0;
		}
		if (average > -1) {
			if (total < 1) {
				n = 0;
				for (i = 0; i < dice; ++i) {
					if (results[i] != 0) {
						printf("%-4d", results[i]);
						++n;
					}
					if (n == 10) {
						printf("\n");
						n = 0;
					}
				}
				if (total == -1) {
					n = plus;
					for (i = 0; i < dice; ++i)
						n += results[i];
					printf("%+4d : %d\n", plus, n);
				} else if (n != 0)
					printf("\n");
			} else {
				n = 0;
				for (i = 0; i < dice; ++i)
					n += results[i];
				n += plus;
				printf("%-4d\n", n);
			}
		}
		if (average) {
			for (i = 0; i < dice; ++i)
				x += results[i];
			x += plus;
		}
	}
	if (average) {
		double foo = (double) x / (double) times;
		printf("%f\n", foo);
	}
	free(results);
}

int
main(int argc, char *argv[])
{
	int i;
	char buf[80];

	srandom((long) (getpid() + time(NULL)));
	if (argc > 1) {
		for (i = 1; i < argc; ++i)
			parse(argv[i]);
	} else {
		/* this strcpy is a workaround */
		strcpy(buf, "l");
		parse(buf);
		while (printf("? "), fgets(buf, 80, stdin)) {
			if (buf[strlen(buf) - 1] == '\n')
				buf[strlen(buf) - 1] = '\0';
			parse(buf);
		}
	}
	return 0;
}
