/* Started Dec 5, 1994 at Xerox.
 * All wrongs reversed. */
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef __USLC__
extern int tgetent(char *, char *);
extern int tgetnum(char *);
#else
#include <curses.h>
#endif

#include "opts.h"
#include "version.h"

/* c - put into columns (default determined by input) */

int columns = 0;
int longest = 0;
int width;
int hmode = 0, nspaces = 2;
int cw = 0;
int vflag = 0;

/*@null@*/
char *buffer = 0;
/*@null@*/
size_t *lines = 0;
int size = 0;
int count = 0;
int lcount = 0;
int lsize = 0;
char *us;
void usage(void);
int spew_output(int);
int slurp_input(int, FILE *);
void callversion(option *, opt_value *);
void csetnum(option *, opt_value *);

void
usage(void) {
	fputs("usage: c [-V] [-n #] [-w #] [-c #] [-h123456789]\n", stderr);
	fputs("\t-n #: use # spaces between columns (default 2)\n", stderr);
	fputs("\t-c #: use # columns (default best fit)\n", stderr);
	fputs("\t-w #: use # characters per line (default stty or 80)\n", stderr);
	fputs("\t-#  : use # columns\n", stderr);
	fputs("\t-h  : adjacent items are left->right, not up->down\n", stderr);
	fputs("\t-V  : print version info\n", stderr);
}

int
spew_output(int nitems) {
	int n = nitems / columns;
	int citem;
	int i, j;

	if (nitems == 0)
		return 0;

	if (n * columns < nitems)
		++n;

	for (i = 0; i < n; ++i) {
		for (j = 0; j < columns; ++j) {
			if (hmode)
				citem = i * columns + j;
			else
				citem = i + (j * n);

			if (citem >= nitems) {
				continue; /* we are done */
			}

			if (j == columns - 1) {
				printf("%s",
					buffer + lines[citem]);
			} else {
				printf("%-*s%*s", cw,
					buffer + lines[citem],
					nspaces, "");
			}
		}

		printf("\n");
	}
	return 0;
}

/* Slurp input.  Our goal is to always leave this function in a state such
 * that count refers (accurately) to the number of lines.  Lines should be
 * an allocated array of count size_t's, offsets into buffer, each `pointing'
 * to a null terminated string.
 * The slurping routine will eat tabs, turning them into multiples-of-8
 * spaces.  Obviously, we need many counters to do this:
 * 1.  Position in line.
 * 2.  Current line.
 * 3.  Position in buffer.
 * 4.  Size of buffer, and of lines[].
 *
 */
int
slurp_input(int llen, FILE *fp) {
	int lpos = 0, i, j, c;

	if (!lines) {
		lines = malloc(256 * sizeof(size_t));
		if (!lines) {
			fprintf(stderr, "%s: couldn't allocate memory.\n", us);
			exit(1);
		}
		lcount = 0;
		lsize = 256;
	}
	if (!buffer) {
		buffer = malloc(4096);
		if (!buffer) {
			fprintf(stderr, "%s: couldn't allocate memory.\n", us);
			exit(1);
		}
		count = 0;
		size = 4096;
	}
	lines[lcount++] = 0;
	while ((c = getc(fp)) != EOF) {
		switch (c) {
		case '\t':
			j = 8 - (lpos % 8);
			if (lpos + j > llen) {
				j = llen - lpos;
			}
			if (count + j > size) {
				buffer = realloc(buffer, size * 2);
				if (!buffer) {
					fprintf(stderr, "%s: couldn't allocate memory.\n", us);
					exit(1);
				}
				size *= 2;
			}
			for (i = 0; i < j; ++i) {
				buffer[count++] = ' ';
			}
			lpos += j;
			c = ' ';
			/* FALLTHROUGH INTENTIONAL */
		default:
			buffer[count++] = (char) c;
			if (lpos++ < llen)
				break;
			
			/* if lpos >= llen, we now eat characters until EOF
			 * or \n, then pretend we just got a newline. */
			while ((c = getc(fp)) != EOF)
				if (c == '\n')
					break;
			--count;
			/*@fallthrough@*/
		case '\n':
			longest = lpos > longest ? lpos : longest;
			lpos = 0;
			buffer[count++] = '\0';
			lines[lcount++] = count;
			if (lcount >= lsize) {
				lines = realloc(lines, lsize * 2 * sizeof(size_t));
				lsize *= 2;
				if (!lines) {
					fprintf(stderr, "%s: couldn't allocate memory.\n", us);
					exit(1);
				}
			}
		}
		if (count >= size) {
			buffer = realloc(buffer, size * 2);
			if (!buffer) {
				fprintf(stderr, "%s: couldn't allocate memory.\n", us);
				exit(1);
			}
			size *= 2;
		}
	}
	buffer[count] = '\0';
#ifdef DEBUG
	fprintf(stderr, "slurped %d/%d chars, %d/%d lines [longest %d]\n",
		count, size, lcount - 1, lsize, longest);
#endif
	return 0;
}

void
callversion(option *op, opt_value *ov) {
	version();
}

void
csetnum(option *op, opt_value *ov) {
	columns = op->abbr - '0';
}

option options[] = {
	{ OTYPE_INT, 'n', "nspaces", "number of spaces",
		OFLAG_DFL, (opt_value *) &nspaces, 0, 0, 0 },
	{ OTYPE_INT, 'w', "width", "output width",
		OFLAG_NONE, (opt_value *) &width, 0, 0, 0 },
	{ OTYPE_NUL, 'h', "horizontal", "horizontal output",
		OFLAG_NONE, (opt_value *) &hmode, 0, 0, 0 },
	{ OTYPE_INT, 'c', "columns", "number of columns",
		OFLAG_NONE, (opt_value *) &columns, 0, 0, 0 },
	{ OTYPE_NUL, '1', 0, "1 column",
		OFLAG_NONE, 0, 0, 0, csetnum },
	{ OTYPE_NUL, '2', 0, "2 columns",
		OFLAG_NONE, 0, 0, 0, csetnum },
	{ OTYPE_NUL, '3', 0, "3 columns",
		OFLAG_NONE, 0, 0, 0, csetnum },
	{ OTYPE_NUL, '4', 0, "4 columns",
		OFLAG_NONE, 0, 0, 0, csetnum },
	{ OTYPE_NUL, '5', 0, "5 columns",
		OFLAG_NONE, 0, 0, 0, csetnum },
	{ OTYPE_NUL, '6', 0, "6 columns",
		OFLAG_NONE, 0, 0, 0, csetnum },
	{ OTYPE_NUL, '7', 0, "7 columns",
		OFLAG_NONE, 0, 0, 0, csetnum },
	{ OTYPE_NUL, '8', 0, "8 columns",
		OFLAG_NONE, 0, 0, 0, csetnum },
	{ OTYPE_NUL, '9', 0, "9 columns",
		OFLAG_NONE, 0, 0, 0, csetnum },
	{ OTYPE_NUL, 'V', "version", "version information",
		OFLAG_NONE, 0, 0, 0, callversion },
	{ OTYPE_END, '\0', 0, 0,
		OFLAG_NONE, 0, 0, 0, 0 },
};

int
main(int argc, char *argv[])
{
	int i, retflag = 0;
	char *s;
	FILE *fp;

	s = argv[0];
	s += strlen(s) - 1;

	if (isdigit(*s))
		columns = *s - '0';
	
	s = argv[0];

	if ((us = strrchr(s, '/')))
		++us;
	else
		us = argv[0];

	{
		char *term;
		char tcp[2048];

		term = getenv("TERM");
		if (term) {
			if (tgetent(tcp, term) > 0) {
				width = tgetnum("co");
			}
		}
	}

	if (optsgets(argc, argv, options)) {
		usage();
		exit(EXIT_FAILURE);
	}

	if (width < 1)
		width = 80;

	if (vflag)
		version();

	if (columns != 0)
		cw = (width - nspaces * (columns - 1)) / columns;
	else
		cw = width;

	if (optsind < argc) {
		for (i = optsind; i < argc; ++i) {
			if (strcmp(argv[i], "-"))
				fp = fopen(argv[i], "r");
			else
				fp = stdin;
			if (!fp) {
				fprintf(stderr, "%s: error: couldn't open file '%s'.\n", us, argv[i]);
				retflag = 1;
			} else
				retflag |= slurp_input(cw, fp);
			if (fp && (fp != stdin))
				fclose(fp);
		}
	} else {
		retflag |= slurp_input(cw, stdin);
	}

	if (columns == 0) {
		cw = longest + nspaces;

		if (cw < nspaces)
			cw = width + nspaces;

		columns = width / cw;
		cw -= nspaces;

		if (columns == 0) {
			columns = 1;
			cw = width;
		}

		while ((columns + 1) * cw + (nspaces * columns) < width)
			++columns;
	}
	return retflag | spew_output(lcount - 1);
}
